//===-- Utility class to test fmod special numbers ------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_TEST_SRC_MATH_FMODTEST_H
#define LLVM_LIBC_TEST_SRC_MATH_FMODTEST_H

#undef LIBC_MATH_USE_SYSTEM_FENV

#include "hdr/errno_macros.h"
#include "hdr/fenv_macros.h"
#include "src/__support/FPUtil/FEnvImpl.h"
#include "test/UnitTest/FEnvSafeTest.h"
#include "test/UnitTest/FPMatcher.h"
#include "test/UnitTest/Test.h"

#ifdef FE_DENORM
#define DENORM_EXCEPT FE_DENORM
#elif defined(__FE_DENORM)
#define DENORM_EXCEPT __FE_DENORM
#else
#define DENORM_EXCEPT 0
#endif // FE_DENORM

#define TEST_SPECIAL(x, y, expected, dom_err, expected_exception)              \
  do {                                                                         \
    LIBC_NAMESPACE::fputil::clear_except(FE_ALL_EXCEPT);                       \
    EXPECT_FP_EQ(expected, f(x, y));                                           \
    EXPECT_MATH_ERRNO((dom_err) ? EDOM : 0);                                   \
    LIBC_NAMESPACE::fputil::clear_except(DENORM_EXCEPT);                       \
    EXPECT_FP_EXCEPTION(expected_exception);                                   \
  } while (0)

#define TEST_REGULAR(x, y, expected) TEST_SPECIAL(x, y, expected, false, 0)

template <typename T>
class FmodTest : public LIBC_NAMESPACE::testing::FEnvSafeTest {

  DECLARE_SPECIAL_CONSTANTS(T)

  static constexpr T one = T(1.0);
  static constexpr T two = T(2.0);
  static constexpr T neg_two = T(-2.0);
  static constexpr T three = T(3.0);
  static constexpr T val_neg_1_1 = T(-1.1);
  static constexpr T val_6_5 = T(6.5);
  static constexpr T val_neg_6_5 = T(-6.5);
  static constexpr T val_2_25 = T(2.25);
  static constexpr T val_neg_2_25 = T(-2.25);

public:
  typedef T (*FModFunc)(T, T);

  void testSpecialNumbers(FModFunc f) {
    // fmod (+0, y) == +0 for y != 0.
    TEST_SPECIAL(zero, three, zero, false, 0);
    TEST_SPECIAL(zero, min_denormal, zero, false, 0);
    TEST_SPECIAL(zero, -min_denormal, zero, false, 0);
    TEST_SPECIAL(zero, min_normal, zero, false, 0);
    TEST_SPECIAL(zero, -min_normal, zero, false, 0);
    TEST_SPECIAL(zero, max_normal, zero, false, 0);
    TEST_SPECIAL(zero, -max_normal, zero, false, 0);

    // fmod (-0, y) == -0 for y != 0.
    TEST_SPECIAL(neg_zero, three, neg_zero, false, 0);
    TEST_SPECIAL(neg_zero, min_denormal, neg_zero, false, 0);
    TEST_SPECIAL(neg_zero, -min_denormal, neg_zero, false, 0);
    TEST_SPECIAL(neg_zero, min_normal, neg_zero, false, 0);
    TEST_SPECIAL(neg_zero, -min_normal, neg_zero, false, 0);
    TEST_SPECIAL(neg_zero, max_normal, neg_zero, false, 0);
    TEST_SPECIAL(neg_zero, -max_normal, neg_zero, false, 0);

    // fmod (+inf, y) == aNaN plus invalid exception.
    TEST_SPECIAL(inf, three, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, val_neg_1_1, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, min_denormal, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, min_normal, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, max_normal, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, inf, aNaN, true, FE_INVALID);
    TEST_SPECIAL(inf, neg_inf, aNaN, true, FE_INVALID);

    // fmod (-inf, y) == aNaN plus invalid exception.
    TEST_SPECIAL(neg_inf, three, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, val_neg_1_1, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, min_denormal, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, min_normal, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, max_normal, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, inf, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_inf, neg_inf, aNaN, true, FE_INVALID);

    // fmod (x, +0) == aNaN plus invalid exception.
    TEST_SPECIAL(three, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(val_neg_1_1, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(zero, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_zero, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(min_denormal, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(min_normal, zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(max_normal, zero, aNaN, true, FE_INVALID);

    // fmod (x, -0) == aNaN plus invalid exception.
    TEST_SPECIAL(three, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(val_neg_1_1, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(zero, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(neg_zero, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(min_denormal, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(min_normal, neg_zero, aNaN, true, FE_INVALID);
    TEST_SPECIAL(max_normal, neg_zero, aNaN, true, FE_INVALID);

    // fmod (x, +inf) == x for x not infinite.
    TEST_SPECIAL(zero, inf, zero, false, 0);
    TEST_SPECIAL(neg_zero, inf, neg_zero, false, 0);
    TEST_SPECIAL(min_denormal, inf, min_denormal, false, 0);
    TEST_SPECIAL(min_normal, inf, min_normal, false, 0);
    TEST_SPECIAL(max_normal, inf, max_normal, false, 0);
    TEST_SPECIAL(three, inf, three, false, 0);
    // fmod (x, -inf) == x for x not infinite.
    TEST_SPECIAL(zero, neg_inf, zero, false, 0);
    TEST_SPECIAL(neg_zero, neg_inf, neg_zero, false, 0);
    TEST_SPECIAL(min_denormal, neg_inf, min_denormal, false, 0);
    TEST_SPECIAL(min_normal, neg_inf, min_normal, false, 0);
    TEST_SPECIAL(max_normal, neg_inf, max_normal, false, 0);
    TEST_SPECIAL(three, neg_inf, three, false, 0);

    TEST_SPECIAL(zero, aNaN, aNaN, false, 0);
    TEST_SPECIAL(zero, neg_aNaN, aNaN, false, 0);
    TEST_SPECIAL(neg_zero, aNaN, aNaN, false, 0);
    TEST_SPECIAL(neg_zero, neg_aNaN, aNaN, false, 0);
    TEST_SPECIAL(one, aNaN, aNaN, false, 0);
    TEST_SPECIAL(one, neg_aNaN, aNaN, false, 0);
    TEST_SPECIAL(inf, aNaN, aNaN, false, 0);
    TEST_SPECIAL(inf, neg_aNaN, aNaN, false, 0);
    TEST_SPECIAL(neg_inf, aNaN, aNaN, false, 0);
    TEST_SPECIAL(neg_inf, neg_aNaN, aNaN, false, 0);
    TEST_SPECIAL(zero, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(zero, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_zero, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_zero, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(one, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(one, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(inf, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(inf, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_inf, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_inf, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(aNaN, zero, aNaN, false, 0);
    TEST_SPECIAL(neg_aNaN, zero, aNaN, false, 0);
    TEST_SPECIAL(aNaN, neg_zero, aNaN, false, 0);
    TEST_SPECIAL(neg_aNaN, neg_zero, aNaN, false, 0);
    TEST_SPECIAL(aNaN, one, aNaN, false, 0);
    TEST_SPECIAL(neg_aNaN, one, aNaN, false, 0);
    TEST_SPECIAL(aNaN, inf, aNaN, false, 0);
    TEST_SPECIAL(neg_aNaN, inf, aNaN, false, 0);
    TEST_SPECIAL(aNaN, neg_inf, aNaN, false, 0);
    TEST_SPECIAL(neg_aNaN, neg_inf, aNaN, false, 0);
    TEST_SPECIAL(sNaN, zero, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, zero, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, neg_zero, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, neg_zero, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, one, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, one, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, inf, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, inf, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, neg_inf, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, neg_inf, aNaN, false, FE_INVALID);
    TEST_SPECIAL(aNaN, aNaN, aNaN, false, 0);
    TEST_SPECIAL(aNaN, neg_aNaN, aNaN, false, 0);
    TEST_SPECIAL(neg_aNaN, aNaN, aNaN, false, 0);
    TEST_SPECIAL(neg_aNaN, neg_aNaN, aNaN, false, 0);
    TEST_SPECIAL(aNaN, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(aNaN, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_aNaN, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_aNaN, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, aNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, neg_aNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, aNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, neg_aNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(sNaN, neg_sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, sNaN, aNaN, false, FE_INVALID);
    TEST_SPECIAL(neg_sNaN, neg_sNaN, aNaN, false, FE_INVALID);

    TEST_SPECIAL(val_6_5, val_2_25, two, false, 0);
    TEST_SPECIAL(val_neg_6_5, val_2_25, neg_two, false, 0);
    TEST_SPECIAL(val_6_5, val_neg_2_25, two, false, 0);
    TEST_SPECIAL(val_neg_6_5, val_neg_2_25, neg_two, false, 0);

    TEST_SPECIAL(max_normal, max_normal, zero, false, 0);
    TEST_SPECIAL(max_normal, -max_normal, zero, false, 0);
    TEST_SPECIAL(max_normal, min_normal, zero, false, 0);
    TEST_SPECIAL(max_normal, -min_normal, zero, false, 0);
    TEST_SPECIAL(max_normal, min_denormal, zero, false, 0);
    TEST_SPECIAL(max_normal, -min_denormal, zero, false, 0);
    TEST_SPECIAL(-max_normal, max_normal, neg_zero, false, 0);
    TEST_SPECIAL(-max_normal, -max_normal, neg_zero, false, 0);
    TEST_SPECIAL(-max_normal, min_normal, neg_zero, false, 0);
    TEST_SPECIAL(-max_normal, -min_normal, neg_zero, false, 0);
    TEST_SPECIAL(-max_normal, min_denormal, neg_zero, false, 0);
    TEST_SPECIAL(-max_normal, -min_denormal, neg_zero, false, 0);

    TEST_SPECIAL(min_normal, max_normal, min_normal, false, 0);
    TEST_SPECIAL(min_normal, -max_normal, min_normal, false, 0);
    TEST_SPECIAL(min_normal, min_normal, zero, false, 0);
    TEST_SPECIAL(min_normal, -min_normal, zero, false, 0);
    TEST_SPECIAL(min_normal, min_denormal, zero, false, 0);
    TEST_SPECIAL(min_normal, -min_denormal, zero, false, 0);
    TEST_SPECIAL(-min_normal, max_normal, -min_normal, false, 0);
    TEST_SPECIAL(-min_normal, -max_normal, -min_normal, false, 0);
    TEST_SPECIAL(-min_normal, min_normal, neg_zero, false, 0);
    TEST_SPECIAL(-min_normal, -min_normal, neg_zero, false, 0);
    TEST_SPECIAL(-min_normal, min_denormal, neg_zero, false, 0);
    TEST_SPECIAL(-min_normal, -min_denormal, neg_zero, false, 0);

    TEST_SPECIAL(min_denormal, max_normal, min_denormal, false, 0);
    TEST_SPECIAL(min_denormal, -max_normal, min_denormal, false, 0);
    TEST_SPECIAL(min_denormal, min_normal, min_denormal, false, 0);
    TEST_SPECIAL(min_denormal, -min_normal, min_denormal, false, 0);
    TEST_SPECIAL(min_denormal, min_denormal, zero, false, 0);
    TEST_SPECIAL(min_denormal, -min_denormal, zero, false, 0);
    TEST_SPECIAL(-min_denormal, max_normal, -min_denormal, false, 0);
    TEST_SPECIAL(-min_denormal, -max_normal, -min_denormal, false, 0);
    TEST_SPECIAL(-min_denormal, min_normal, -min_denormal, false, 0);
    TEST_SPECIAL(-min_denormal, -min_normal, -min_denormal, false, 0);
    TEST_SPECIAL(-min_denormal, min_denormal, neg_zero, false, 0);
    TEST_SPECIAL(-min_denormal, -min_denormal, neg_zero, false, 0);
  }

  void testRegularExtreme(FModFunc f) {
    if constexpr (sizeof(T) < sizeof(float))
      return;
    TEST_REGULAR(T(0x1p127), T(0x3p-149), T(0x1p-149));
    TEST_REGULAR(T(0x1p127), T(-0x3p-149), T(0x1p-149));
    TEST_REGULAR(T(0x1p127), T(0x3p-148), T(0x1p-147));
    TEST_REGULAR(T(0x1p127), T(-0x3p-148), T(0x1p-147));
    TEST_REGULAR(T(0x1p127), T(0x3p-126), T(0x1p-125));
    TEST_REGULAR(T(0x1p127), T(-0x3p-126), T(0x1p-125));
    TEST_REGULAR(T(-0x1p127), T(0x3p-149), T(-0x1p-149));
    TEST_REGULAR(T(-0x1p127), T(-0x3p-149), T(-0x1p-149));
    TEST_REGULAR(T(-0x1p127), T(0x3p-148), T(-0x1p-147));
    TEST_REGULAR(T(-0x1p127), T(-0x3p-148), T(-0x1p-147));
    TEST_REGULAR(T(-0x1p127), T(0x3p-126), T(-0x1p-125));
    TEST_REGULAR(T(-0x1p127), T(-0x3p-126), T(-0x1p-125));

    if constexpr (sizeof(T) < sizeof(double))
      return;
    TEST_REGULAR(T(0x1p1023), T(0x3p-1074), T(0x1p-1073));
    TEST_REGULAR(T(0x1p1023), T(-0x3p-1074), T(0x1p-1073));
    TEST_REGULAR(T(0x1p1023), T(0x3p-1073), T(0x1p-1073));
    TEST_REGULAR(T(0x1p1023), T(-0x3p-1073), T(0x1p-1073));
    TEST_REGULAR(T(0x1p1023), T(0x3p-1022), T(0x1p-1021));
    TEST_REGULAR(T(0x1p1023), T(-0x3p-1022), T(0x1p-1021));
    TEST_REGULAR(T(-0x1p1023), T(0x3p-1074), T(-0x1p-1073));
    TEST_REGULAR(T(-0x1p1023), T(-0x3p-1074), T(-0x1p-1073));
    TEST_REGULAR(T(-0x1p1023), T(0x3p-1073), T(-0x1p-1073));
    TEST_REGULAR(T(-0x1p1023), T(-0x3p-1073), T(-0x1p-1073));
    TEST_REGULAR(T(-0x1p1023), T(0x3p-1022), T(-0x1p-1021));
    TEST_REGULAR(T(-0x1p1023), T(-0x3p-1022), T(-0x1p-1021));
  }
};

#define LIST_FMOD_TESTS(T, func)                                               \
  using LlvmLibcFmodTest = FmodTest<T>;                                        \
  TEST_F(LlvmLibcFmodTest, SpecialNumbers) { testSpecialNumbers(&func); }      \
  TEST_F(LlvmLibcFmodTest, RegularExtreme) { testRegularExtreme(&func); }

#endif // LLVM_LIBC_TEST_SRC_MATH_FMODTEST_H
